<?php

declare(strict_types=1);

require_once __DIR__ . '/../company/company_repo.php';
require_once APP_PATH . '/helpers.php';

const VOUCHER_TYPES = ['JOURNAL', 'PAYMENT', 'RECEIPT', 'CONTRA', 'SALES', 'PURCHASE', 'CREDIT_NOTE', 'DEBIT_NOTE', 'OPENING', 'REVERSAL'];

function validate_balanced(array $lines): bool
{
    $totalDr = '0';
    $totalCr = '0';
    foreach ($lines as $line) {
        $dr = (string) ($line['debit'] ?? 0);
        $cr = (string) ($line['credit'] ?? 0);
        $totalDr = decimal_add($totalDr, $dr);
        $totalCr = decimal_add($totalCr, $cr);
    }
    return decimal_compare($totalDr, $totalCr) === 0;
}

function post_voucher(
    int $companyId,
    int $fyId,
    string $type,
    string $date,
    ?string $narration,
    array $lines,
    ?string $refModule = null,
    ?int $refId = null
): int {
    $pdo = db();

    // Validate date within FY
    $fy = company_repo_fy_find($fyId);
    if (!$fy || $fy['company_id'] != $companyId) {
        throw new InvalidArgumentException('Invalid company or FY.');
    }
    if ($date < $fy['start_date'] || $date > $fy['end_date']) {
        throw new InvalidArgumentException('Voucher date must be within financial year.');
    }

    // Check period lock
    if (company_repo_is_locked($fyId, $date, can_override_lock())) {
        throw new InvalidArgumentException('Cannot post: period is locked.');
    }

    // Validate balanced
    if (!validate_balanced($lines)) {
        throw new InvalidArgumentException('Debits must equal credits.');
    }

    // Validate ledgers exist and are active
    foreach ($lines as $line) {
        $ledgerId = (int) ($line['ledger_id'] ?? 0);
        $dr = (float) ($line['debit'] ?? 0);
        $cr = (float) ($line['credit'] ?? 0);
        if ($ledgerId <= 0) {
            throw new InvalidArgumentException('Invalid ledger in voucher line.');
        }
        if ($dr < 0 || $cr < 0) {
            throw new InvalidArgumentException('Debit/credit cannot be negative.');
        }
        if ($dr == 0 && $cr == 0) {
            throw new InvalidArgumentException('Each line must have debit or credit.');
        }
        $stmt = $pdo->prepare('SELECT 1 FROM ledgers WHERE id = ? AND company_id = ? AND is_active = 1');
        $stmt->execute([$ledgerId, $companyId]);
        if (!$stmt->fetch()) {
            throw new InvalidArgumentException('Ledger not found or inactive.');
        }
    }

    $pdo->beginTransaction();
    try {
        // Generate voucher_no with row lock
        $voucherNo = vouchers_repo_next_number($companyId, $fyId, $type);

        $stmt = $pdo->prepare('
            INSERT INTO vouchers (company_id, financial_year_id, voucher_no, voucher_type, voucher_date, narration, ref_module, ref_id, posted_by, posted_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ');
        $stmt->execute([$companyId, $fyId, $voucherNo, $type, $date, $narration, $refModule, $refId, auth_id()]);
        $voucherId = (int) $pdo->lastInsertId();

        $lineStmt = $pdo->prepare('INSERT INTO voucher_lines (voucher_id, company_id, ledger_id, debit, credit, narration, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?)');
        foreach ($lines as $i => $line) {
            $lineStmt->execute([
                $voucherId,
                $companyId,
                $line['ledger_id'],
                $line['debit'] ?? 0,
                $line['credit'] ?? 0,
                $line['narration'] ?? null,
                $i,
            ]);
        }

        audit_log('INSERT', 'vouchers', $voucherId, null, ['voucher_no' => $voucherNo, 'type' => $type, 'date' => $date]);

        $pdo->commit();
        return $voucherId;
    } catch (Throwable $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function void_voucher(int $voucherId, string $reason): void
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT * FROM vouchers WHERE id = ? AND is_deleted = 0');
    $stmt->execute([$voucherId]);
    $v = $stmt->fetch();
    if (!$v) {
        throw new InvalidArgumentException('Voucher not found or already voided.');
    }

    if (company_repo_is_locked((int) $v['financial_year_id'], $v['voucher_date'], can_override_lock())) {
        throw new InvalidArgumentException('Cannot void: period is locked.');
    }

    $pdo->beginTransaction();
    try {
        $pdo->prepare('UPDATE vouchers SET is_deleted = 1, void_reason = ? WHERE id = ?')->execute([$reason, $voucherId]);
        audit_log('VOID', 'vouchers', $voucherId, null, ['reason' => $reason]);
        $pdo->commit();
    } catch (Throwable $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function audit_log(string $action, ?string $tableName, ?int $recordId, ?array $before, ?array $after): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO audit_logs (user_id, action, table_name, record_id, before_data, after_data, ip_address, user_agent) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([
        auth_id(),
        $action,
        $tableName,
        $recordId,
        $before ? json_encode($before) : null,
        $after ? json_encode($after) : null,
        $_SERVER['REMOTE_ADDR'] ?? null,
        $_SERVER['HTTP_USER_AGENT'] ?? null,
    ]);
}
