<?php

declare(strict_types=1);

require_once __DIR__ . '/users_service.php';
require_once __DIR__ . '/users_repo.php';

function users_list_page(): void
{
    require_permission('admin.users');
    $users = users_repo_list();
    $pageTitle = 'Users';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Users</h1>
    <a href="?r=admin/user/create" class="btn btn-primary">Add User</a>
    <table class="data-table">
        <thead>
            <tr><th>Username</th><th>Name</th><th>Roles</th><th>Active</th></tr>
        </thead>
        <tbody>
            <?php foreach ($users as $u): ?>
            <tr>
                <td><?= e($u['username']) ?></td>
                <td><?= e($u['name'] ?? '') ?></td>
                <td><?= e($u['roles'] ?? '-') ?></td>
                <td><?= $u['is_active'] ? 'Yes' : 'No' ?></td>
                <td><a href="?r=admin/user/edit&id=<?= $u['id'] ?>">Edit</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function user_create_page(): void
{
    require_permission('admin.users');
    $roles = users_repo_roles_list();
    $pageTitle = 'Add User';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Add User</h1>
    <form method="post" action="?r=admin/user/save">
        <?= csrf_field() ?>
        <div class="form-group">
            <label for="username">Username *</label>
            <input type="text" id="username" name="username" required value="<?= e(old('username')) ?>">
        </div>
        <div class="form-group">
            <label for="password">Password *</label>
            <input type="password" id="password" name="password" required>
        </div>
        <div class="form-group">
            <label for="name">Name</label>
            <input type="text" id="name" name="name" value="<?= e(old('name')) ?>">
        </div>
        <div class="form-group">
            <label for="email">Email</label>
            <input type="email" id="email" name="email" value="<?= e(old('email')) ?>">
        </div>
        <div class="form-group">
            <label>Roles</label>
            <?php foreach ($roles as $r): ?>
            <label><input type="checkbox" name="roles[]" value="<?= $r['id'] ?>"> <?= e($r['name']) ?></label>
            <?php endforeach; ?>
        </div>
        <div class="form-group">
            <label><input type="checkbox" name="is_active" value="1" checked> Active</label>
        </div>
        <button type="submit" class="btn btn-primary">Save</button>
        <a href="?r=admin/users" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function user_save_action(): void
{
    csrf_verify();
    require_permission('admin.users');

    $data = [
        'username' => trim($_POST['username'] ?? ''),
        'password' => $_POST['password'] ?? '',
        'name' => trim($_POST['name'] ?? ''),
        'email' => trim($_POST['email'] ?? ''),
        'is_active' => isset($_POST['is_active']) ? 1 : 0,
        'roles' => $_POST['roles'] ?? [],
    ];

    try {
        users_service_create($data);
        flash_set('success', 'User created.');
        redirect_route('admin/users');
    } catch (Throwable $e) {
        $_SESSION['_old'] = $data;
        flash_set('error', $e->getMessage());
        redirect_route('admin/user/create');
    }
}

function roles_list_page(): void
{
    require_permission('admin.roles');
    $pageTitle = 'Roles';
    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Roles</h1>
    <p>Manage roles in database. Admin role has full access.</p>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function audit_log_page(): void
{
    require_permission('admin.audit');
    $pdo = db();
    $page = max(1, (int) ($_GET['page'] ?? 1));
    $perPage = 50;
    $offset = ($page - 1) * $perPage;

    $stmt = $pdo->prepare('SELECT COUNT(*) FROM audit_logs');
    $stmt->execute();
    $total = (int) $stmt->fetchColumn();
    $totalPages = (int) ceil($total / $perPage);

    $stmt = $pdo->prepare('SELECT al.*, u.username FROM audit_logs al LEFT JOIN users u ON u.id = al.user_id ORDER BY al.created_at DESC LIMIT ? OFFSET ?');
    $stmt->execute([$perPage, $offset]);
    $logs = $stmt->fetchAll();

    $pageTitle = 'Audit Log';
    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Audit Log</h1>
    <table class="data-table">
        <thead>
            <tr><th>Time</th><th>User</th><th>Action</th><th>Table</th><th>Record ID</th></tr>
        </thead>
        <tbody>
            <?php foreach ($logs as $l): ?>
            <tr>
                <td><?= e($l['created_at']) ?></td>
                <td><?= e($l['username'] ?? '-') ?></td>
                <td><?= e($l['action']) ?></td>
                <td><?= e($l['table_name'] ?? '-') ?></td>
                <td><?= e($l['record_id'] ?? '-') ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php if ($totalPages > 1): ?>
    <div class="pagination">
        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
        <a href="?r=admin/audit&page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
        <?php endfor; ?>
    </div>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}
