<?php

declare(strict_types=1);

function company_repo_list(): array
{
    $pdo = db();
    $stmt = $pdo->query('SELECT * FROM companies WHERE is_active = 1 ORDER BY name');
    return $stmt->fetchAll();
}

function company_repo_find(int $id): ?array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT * FROM companies WHERE id = ?');
    $stmt->execute([$id]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function company_repo_create(array $data): int
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO companies (name, address, tax_id, phone, email) VALUES (?, ?, ?, ?, ?)');
    $stmt->execute([
        $data['name'],
        $data['address'] ?? '',
        $data['tax_id'] ?? '',
        $data['phone'] ?? '',
        $data['email'] ?? '',
    ]);
    return (int) $pdo->lastInsertId();
}

function company_repo_update(int $id, array $data): void
{
    $pdo = db();
    $stmt = $pdo->prepare('UPDATE companies SET name=?, address=?, tax_id=?, phone=?, email=? WHERE id=?');
    $stmt->execute([
        $data['name'],
        $data['address'] ?? '',
        $data['tax_id'] ?? '',
        $data['phone'] ?? '',
        $data['email'] ?? '',
        $id,
    ]);
}

function company_repo_fy_list(int $companyId): array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT fy.* FROM financial_years fy WHERE fy.company_id = ? ORDER BY fy.start_date DESC');
    $stmt->execute([$companyId]);
    return $stmt->fetchAll();
}

function company_repo_fy_find(int $id): ?array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT fy.*, c.name as company_name FROM financial_years fy JOIN companies c ON c.id = fy.company_id WHERE fy.id = ?');
    $stmt->execute([$id]);
    return $stmt->fetch() ?: null;
}

function company_repo_fy_create(array $data): int
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO financial_years (company_id, name, start_date, end_date) VALUES (?, ?, ?, ?)');
    $stmt->execute([
        $data['company_id'],
        $data['name'],
        $data['start_date'],
        $data['end_date'],
    ]);
    return (int) $pdo->lastInsertId();
}

function company_repo_lock_list(int $fyId): array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT * FROM period_locks WHERE financial_year_id = ? ORDER BY lock_date');
    $stmt->execute([$fyId]);
    return $stmt->fetchAll();
}

function company_repo_lock_save(int $fyId, string $lockDate): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT IGNORE INTO period_locks (financial_year_id, lock_date, created_by) VALUES (?, ?, ?)');
    $stmt->execute([$fyId, $lockDate, auth_id()]);
}

function company_repo_is_locked(int $fyId, string $date, bool $allowOverride = false): bool
{
    if ($allowOverride && can_override_lock()) {
        return false;
    }
    $pdo = db();
    $stmt = $pdo->prepare('SELECT 1 FROM period_locks WHERE financial_year_id = ? AND lock_date >= ? LIMIT 1');
    $stmt->execute([$fyId, $date]);
    return (bool) $stmt->fetch();
}

function company_repo_fy_in_range(int $fyId, string $date): ?array
{
    $fy = company_repo_fy_find($fyId);
    if (!$fy) return null;
    if ($date >= $fy['start_date'] && $date <= $fy['end_date']) {
        return $fy;
    }
    return null;
}
