<?php

declare(strict_types=1);

function masters_repo_groups_list(int $companyId): array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT ag.*, (SELECT COUNT(*) FROM ledgers l WHERE l.group_id = ag.id AND l.is_active = 1) as ledger_count FROM account_groups ag WHERE ag.company_id = ? ORDER BY ag.sort_order, ag.name');
    $stmt->execute([$companyId]);
    return $stmt->fetchAll();
}

function masters_repo_group_find(int $id): ?array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT * FROM account_groups WHERE id = ?');
    $stmt->execute([$id]);
    return $stmt->fetch() ?: null;
}

function masters_repo_group_create(array $data): int
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO account_groups (company_id, parent_id, name, nature, sort_order) VALUES (?, ?, ?, ?, ?)');
    $stmt->execute([
        $data['company_id'],
        $data['parent_id'] ?? null,
        $data['name'],
        $data['nature'],
        $data['sort_order'] ?? 0,
    ]);
    return (int) $pdo->lastInsertId();
}

function masters_repo_ledgers_list(int $companyId, ?int $groupId = null): array
{
    $pdo = db();
    if ($groupId) {
        $stmt = $pdo->prepare('SELECT l.*, ag.name as group_name FROM ledgers l JOIN account_groups ag ON ag.id = l.group_id WHERE l.company_id = ? AND l.group_id = ? ORDER BY l.name');
        $stmt->execute([$companyId, $groupId]);
    } else {
        $stmt = $pdo->prepare('SELECT l.*, ag.name as group_name FROM ledgers l JOIN account_groups ag ON ag.id = l.group_id WHERE l.company_id = ? ORDER BY ag.sort_order, l.name');
        $stmt->execute([$companyId]);
    }
    return $stmt->fetchAll();
}

function masters_repo_ledger_find(int $id): ?array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT l.*, ag.name as group_name, ag.nature FROM ledgers l JOIN account_groups ag ON ag.id = l.group_id WHERE l.id = ?');
    $stmt->execute([$id]);
    return $stmt->fetch() ?: null;
}

function masters_repo_ledger_create(array $data): int
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO ledgers (company_id, group_id, name, code, ledger_type, opening_balance_dc, opening_balance, is_active) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([
        $data['company_id'],
        $data['group_id'],
        $data['name'],
        $data['code'] ?? null,
        $data['ledger_type'] ?? 'GENERAL',
        $data['opening_balance_dc'] ?? 'D',
        $data['opening_balance'] ?? 0,
        $data['is_active'] ?? 1,
    ]);
    return (int) $pdo->lastInsertId();
}

function masters_repo_ledger_update(int $id, array $data): void
{
    $pdo = db();
    $stmt = $pdo->prepare('UPDATE ledgers SET name=?, code=?, ledger_type=?, group_id=?, is_active=? WHERE id=?');
    $stmt->execute([
        $data['name'],
        $data['code'] ?? null,
        $data['ledger_type'] ?? 'GENERAL',
        $data['group_id'],
        $data['is_active'] ?? 1,
        $id,
    ]);
}

function masters_repo_openings_list(int $companyId, int $fyId): array
{
    $pdo = db();
    $stmt = $pdo->prepare("
        SELECT l.id, l.name, l.code, ag.name as group_name,
               COALESCE(lo.opening_debit, 0) as opening_debit,
               COALESCE(lo.opening_credit, 0) as opening_credit
        FROM ledgers l
        JOIN account_groups ag ON ag.id = l.group_id
        LEFT JOIN ledger_openings lo ON lo.ledger_id = l.id AND lo.financial_year_id = ?
        WHERE l.company_id = ? AND l.is_active = 1
        ORDER BY ag.sort_order, l.name
    ");
    $stmt->execute([$fyId, $companyId]);
    return $stmt->fetchAll();
}

function masters_repo_openings_save(int $fyId, array $openings): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO ledger_openings (ledger_id, financial_year_id, opening_debit, opening_credit) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE opening_debit = VALUES(opening_debit), opening_credit = VALUES(opening_credit)');
    foreach ($openings as $o) {
        $stmt->execute([
            $o['ledger_id'],
            $fyId,
            $o['opening_debit'] ?? 0,
            $o['opening_credit'] ?? 0,
        ]);
    }
}
