<?php

declare(strict_types=1);

require_once __DIR__ . '/purchase_repo.php';
require_once __DIR__ . '/../accounts/posting_engine.php';

function purchase_service_post_bill(int $billId): int
{
    $bill = purchase_repo_find($billId);
    if (!$bill) {
        throw new InvalidArgumentException('Bill not found.');
    }
    if ($bill['status'] === 'POSTED') {
        throw new InvalidArgumentException('Bill already posted.');
    }

    $lines = purchase_service_build_voucher_lines($bill);

    $voucherId = post_voucher(
        (int) $bill['company_id'],
        (int) $bill['financial_year_id'],
        'PURCHASE',
        $bill['bill_date'],
        'Purchase Bill ' . $bill['bill_no'],
        $lines,
        'purchase/bill',
        $billId
    );

    purchase_repo_update_voucher($billId, $voucherId);
    open_refs_repo_create_ap(
        (int) $bill['company_id'],
        (int) $bill['supplier_ledger_id'],
        'purchase_bill',
        $billId,
        $bill['bill_date'],
        $bill['bill_no'],
        (float) $bill['grand_total']
    );

    return $voucherId;
}

function purchase_service_build_voucher_lines(array $bill): array
{
    $billLines = purchase_repo_lines((int) $bill['id']);
    $taxLines = purchase_repo_taxes((int) $bill['id']);

    $lines = [];
    $grandTotal = (float) $bill['grand_total'];
    $subtotal = (float) $bill['subtotal'];
    $discount = (float) $bill['discount_amount'];
    $charges = (float) $bill['charges_amount'];
    $purchaseDebit = $subtotal - $discount;

    $lines[] = [
        'ledger_id' => $bill['supplier_ledger_id'],
        'debit' => 0,
        'credit' => $grandTotal,
        'narration' => 'Purchase Bill ' . $bill['bill_no'],
    ];

    foreach ($billLines as $bl) {
        $ledgerId = $bl['ledger_id'] ?? null;
        if (!$ledgerId) continue;
        $amount = (float) $bl['amount'];
        $lines[] = ['ledger_id' => $ledgerId, 'debit' => $amount, 'credit' => 0, 'narration' => $bl['description'] ?? ''];
        $purchaseDebit -= $amount;
    }
    if ($purchaseDebit > 0) {
        $defaultPurchaseLedger = purchase_service_get_default_purchase_ledger((int) $bill['company_id']);
        if ($defaultPurchaseLedger) {
            $lines[] = ['ledger_id' => $defaultPurchaseLedger, 'debit' => $purchaseDebit, 'credit' => 0, 'narration' => 'Purchase'];
        }
    }

    foreach ($taxLines as $t) {
        $ledgerId = $t['ledger_id'] ?? null;
        if ($ledgerId && (float) $t['amount'] > 0) {
            $lines[] = ['ledger_id' => $ledgerId, 'debit' => (float) $t['amount'], 'credit' => 0, 'narration' => 'Input Tax'];
        }
    }

    if ($charges > 0) {
        $chargesLedger = purchase_service_get_charges_ledger((int) $bill['company_id']);
        if ($chargesLedger) {
            $lines[] = ['ledger_id' => $chargesLedger, 'debit' => $charges, 'credit' => 0, 'narration' => 'Charges'];
        }
    }

    return $lines;
}

function purchase_service_get_default_purchase_ledger(int $companyId): ?int
{
    $pdo = db();
    $stmt = $pdo->prepare("SELECT id FROM ledgers WHERE company_id = ? AND is_active = 1 AND group_id IN (SELECT id FROM account_groups WHERE company_id = ? AND nature = 'EXPENSE') LIMIT 1");
    $stmt->execute([$companyId, $companyId]);
    $row = $stmt->fetch();
    return $row ? (int) $row['id'] : null;
}

function purchase_service_get_charges_ledger(int $companyId): ?int
{
    $pdo = db();
    $stmt = $pdo->prepare("SELECT id FROM ledgers WHERE company_id = ? AND is_active = 1 LIMIT 1");
    $stmt->execute([$companyId]);
    $row = $stmt->fetch();
    return $row ? (int) $row['id'] : null;
}
