<?php

declare(strict_types=1);

require_once __DIR__ . '/reports_service.php';
require_once __DIR__ . '/reports_repo.php';
require_once APP_PATH . '/features/company/company_service.php';

function reports_trial_balance_page(): void
{
    $ctx = company_service_require_context();
    $asOf = $_GET['as_of'] ?? null;
    $result = reports_service_trial_balance($ctx['company_id'], $ctx['fy_id'], $asOf);
    $pageTitle = 'Trial Balance';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Trial Balance</h1>
    <form method="get" class="inline-form">
        <input type="hidden" name="r" value="reports/trial_balance">
        <label>As of: <input type="date" name="as_of" value="<?= e($asOf ?? $ctx['fy']['end_date']) ?>"></label>
        <button type="submit" class="btn btn-primary">Refresh</button>
    </form>
    <table class="data-table">
        <thead>
            <tr><th>Ledger</th><th class="text-right">Opening Dr</th><th class="text-right">Opening Cr</th><th class="text-right">Period Dr</th><th class="text-right">Period Cr</th><th class="text-right">Closing Dr</th><th class="text-right">Closing Cr</th><th></th></tr>
        </thead>
        <tbody>
            <?php foreach ($result['rows'] as $r): ?>
            <tr>
                <td><?= e($r['name']) ?> <?= $r['code'] ? '(' . e($r['code']) . ')' : '' ?></td>
                <td class="text-right"><?= $r['opening_debit'] > 0 ? format_money((float) $r['opening_debit']) : '' ?></td>
                <td class="text-right"><?= $r['opening_credit'] > 0 ? format_money((float) $r['opening_credit']) : '' ?></td>
                <td class="text-right"><?= $r['period_debit'] > 0 ? format_money((float) $r['period_debit']) : '' ?></td>
                <td class="text-right"><?= $r['period_credit'] > 0 ? format_money((float) $r['period_credit']) : '' ?></td>
                <td class="text-right"><?= $r['closing_debit'] > 0 ? format_money((float) $r['closing_debit']) : '' ?></td>
                <td class="text-right"><?= $r['closing_credit'] > 0 ? format_money((float) $r['closing_credit']) : '' ?></td>
                <td><a href="?r=reports/ledger&ledger_id=<?= $r['id'] ?>&from=<?= e($ctx['fy']['start_date']) ?>&to=<?= e($asOf ?? $ctx['fy']['end_date']) ?>">Ledger</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr><th>Total</th><th class="text-right"></th><th class="text-right"></th><th class="text-right"></th><th class="text-right"></th><th class="text-right"><?= format_money((float) $result['total_dr']) ?></th><th class="text-right"><?= format_money((float) $result['total_cr']) ?></th><th></th></tr>
        </tfoot>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function reports_ledger_page(): void
{
    $ctx = company_service_require_context();
    $ledgerId = (int) ($_GET['ledger_id'] ?? 0);
    $from = $_GET['from'] ?? $ctx['fy']['start_date'];
    $to = $_GET['to'] ?? $ctx['fy']['end_date'];
    $data = reports_repo_ledger_statement($ctx['company_id'], $ctx['fy_id'], $ledgerId, $from, $to);
    if (!$data['ledger']) {
        flash_set('error', 'Ledger not found.');
        redirect_route('reports/trial_balance');
    }
    $pageTitle = 'Ledger - ' . $data['ledger']['name'];

    $balance = (float) $data['opening']['debit'] - (float) $data['opening']['credit'];
    foreach ($data['lines'] as $l) {
        $balance += (float) $l['debit'] - (float) $l['credit'];
    }

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Ledger: <?= e($data['ledger']['name']) ?></h1>
    <p>Period: <?= format_date($from) ?> to <?= format_date($to) ?></p>
    <form method="get" class="inline-form">
        <input type="hidden" name="r" value="reports/ledger">
        <input type="hidden" name="ledger_id" value="<?= $ledgerId ?>">
        <input type="date" name="from" value="<?= e($from) ?>"> to <input type="date" name="to" value="<?= e($to) ?>">
        <button type="submit" class="btn btn-primary">Refresh</button>
    </form>
    <table class="data-table">
        <thead>
            <tr><th>Date</th><th>Voucher</th><th>Type</th><th class="text-right">Debit</th><th class="text-right">Credit</th><th class="text-right">Balance</th><th>Narration</th><th></th></tr>
        </thead>
        <tbody>
            <?php
            $running = (float) $data['opening']['debit'] - (float) $data['opening']['credit'];
            if ($running != 0):
            ?>
            <tr>
                <td><?= format_date($from) ?></td>
                <td>-</td>
                <td>Opening</td>
                <td class="text-right"><?= $data['opening']['debit'] > 0 ? format_money((float) $data['opening']['debit']) : '' ?></td>
                <td class="text-right"><?= $data['opening']['credit'] > 0 ? format_money((float) $data['opening']['credit']) : '' ?></td>
                <td class="text-right"><?= format_money($running) ?> <?= $running >= 0 ? 'Dr' : 'Cr' ?></td>
                <td></td>
                <td></td>
            </tr>
            <?php endif;
            foreach ($data['lines'] as $l):
                $running += (float) $l['debit'] - (float) $l['credit'];
            ?>
            <tr>
                <td><?= format_date($l['voucher_date']) ?></td>
                <td><?= e($l['voucher_no']) ?></td>
                <td><?= e($l['voucher_type']) ?></td>
                <td class="text-right"><?= $l['debit'] > 0 ? format_money((float) $l['debit']) : '' ?></td>
                <td class="text-right"><?= $l['credit'] > 0 ? format_money((float) $l['credit']) : '' ?></td>
                <td class="text-right"><?= format_money($running) ?> <?= $running >= 0 ? 'Dr' : 'Cr' ?></td>
                <td><?= e($l['narration'] ?? '') ?></td>
                <td><a href="?r=accounts/voucher/view&id=<?= $l['voucher_id'] ?>">View</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr><th colspan="5">Closing Balance</th><th class="text-right"><?= format_money($balance) ?> <?= $balance >= 0 ? 'Dr' : 'Cr' ?></th><th colspan="2"></th></tr>
        </tfoot>
    </table>
    <a href="?r=reports/trial_balance" class="btn btn-secondary">Back to Trial Balance</a>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function reports_pl_page(): void
{
    $ctx = company_service_require_context();
    $from = $_GET['from'] ?? $ctx['fy']['start_date'];
    $to = $_GET['to'] ?? $ctx['fy']['end_date'];
    $rows = reports_repo_pl($ctx['company_id'], $ctx['fy_id'], $from, $to);
    $pageTitle = 'Profit & Loss';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Profit & Loss</h1>
    <p>Period: <?= format_date($from) ?> - <?= format_date($to) ?></p>
    <table class="data-table">
        <thead><tr><th>Ledger</th><th>Group</th><th class="text-right">Amount</th></tr></thead>
        <tbody>
            <?php
            $incomeTotal = 0;
            $expenseTotal = 0;
            foreach ($rows as $r):
                $net = (float) $r['opening_net'] + (float) $r['period_net'];
                if ($r['nature'] === 'INCOME') $incomeTotal += $net;
                else $expenseTotal += $net;
            ?>
            <tr><td><?= e($r['ledger_name']) ?></td><td><?= e($r['group_name']) ?></td><td class="text-right"><?= format_money($net) ?></td></tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr><th>Total Income</th><th></th><th class="text-right"><?= format_money($incomeTotal) ?></th></tr>
            <tr><th>Total Expense</th><th></th><th class="text-right"><?= format_money($expenseTotal) ?></th></tr>
            <tr><th>Net Profit</th><th></th><th class="text-right"><?= format_money($incomeTotal - $expenseTotal) ?></th></tr>
        </tfoot>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function reports_bs_page(): void
{
    $ctx = company_service_require_context();
    $asOf = $_GET['as_of'] ?? $ctx['fy']['end_date'];
    $rows = reports_repo_bs($ctx['company_id'], $ctx['fy_id'], $asOf);
    $pageTitle = 'Balance Sheet';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Balance Sheet</h1>
    <p>As of: <?= format_date($asOf) ?></p>
    <table class="data-table">
        <thead><tr><th>Ledger</th><th>Group</th><th class="text-right">Amount</th></tr></thead>
        <tbody>
            <?php
            $assetTotal = 0;
            $liabTotal = 0;
            foreach ($rows as $r):
                $net = (float) $r['opening_net'] + (float) $r['period_net'];
                if ($r['nature'] === 'ASSET') $assetTotal += $net;
                else $liabTotal += abs($net);
            ?>
            <tr><td><?= e($r['ledger_name']) ?></td><td><?= e($r['group_name']) ?></td><td class="text-right"><?= format_money($net) ?></td></tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr><th>Total Assets</th><th></th><th class="text-right"><?= format_money($assetTotal) ?></th></tr>
            <tr><th>Total Liabilities</th><th></th><th class="text-right"><?= format_money($liabTotal) ?></th></tr>
        </tfoot>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function reports_daybook_page(): void
{
    $ctx = company_service_require_context();
    $date = $_GET['date'] ?? date('Y-m-d');
    $page = max(1, (int) ($_GET['page'] ?? 1));
    $config = config();
    $perPage = $config['pagination']['per_page'] ?? 50;
    $result = reports_repo_daybook($ctx['company_id'], $ctx['fy_id'], $date, $page, $perPage);
    $totalPages = (int) ceil($result['total'] / $perPage);
    $pageTitle = 'Daybook';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Daybook - <?= format_date($date) ?></h1>
    <form method="get"><input type="hidden" name="r" value="reports/daybook"><input type="date" name="date" value="<?= e($date) ?>"><button type="submit" class="btn btn-primary">Go</button></form>
    <table class="data-table">
        <thead><tr><th>Voucher No</th><th>Type</th><th>Narration</th><th></th></tr></thead>
        <tbody>
            <?php foreach ($result['items'] as $v): ?>
            <tr><td><?= e($v['voucher_no']) ?></td><td><?= e($v['voucher_type']) ?></td><td><?= e(substr($v['narration'] ?? '', 0, 60)) ?></td><td><a href="?r=accounts/voucher/view&id=<?= $v['id'] ?>">View</a></td></tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php if ($totalPages > 1): ?>
    <div class="pagination">
        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
        <a href="?r=reports/daybook&date=<?= e($date) ?>&page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
        <?php endfor; ?>
    </div>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function reports_outstanding_page(): void
{
    $ctx = company_service_require_context();
    $pageTitle = 'Outstanding';
    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Outstanding Ageing</h1>
    <p>Outstanding report - requires sales/purchase data.</p>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function reports_sales_register_page(): void
{
    $ctx = company_service_require_context();
    $pageTitle = 'Sales Register';
    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Sales Register</h1>
    <p>Sales register report.</p>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function reports_purchase_register_page(): void
{
    $ctx = company_service_require_context();
    $pageTitle = 'Purchase Register';
    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Purchase Register</h1>
    <p>Purchase register report.</p>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}
