<?php

declare(strict_types=1);

function sales_repo_list(int $companyId, int $fyId, int $page = 1, int $perPage = 20): array
{
    $pdo = db();
    $offset = ($page - 1) * $perPage;

    $stmt = $pdo->prepare('SELECT COUNT(*) FROM sales_invoices WHERE company_id = ? AND financial_year_id = ?');
    $stmt->execute([$companyId, $fyId]);
    $total = (int) $stmt->fetchColumn();

    $stmt = $pdo->prepare('
        SELECT si.*, l.name as customer_name
        FROM sales_invoices si
        JOIN ledgers l ON l.id = si.customer_ledger_id
        WHERE si.company_id = ? AND si.financial_year_id = ?
        ORDER BY si.invoice_date DESC, si.id DESC
        LIMIT ? OFFSET ?
    ');
    $stmt->execute([$companyId, $fyId, $perPage, $offset]);
    return ['items' => $stmt->fetchAll(), 'total' => $total];
}

function sales_repo_find(int $id): ?array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT si.*, l.name as customer_name FROM sales_invoices si JOIN ledgers l ON l.id = si.customer_ledger_id WHERE si.id = ?');
    $stmt->execute([$id]);
    return $stmt->fetch() ?: null;
}

function sales_repo_lines(int $invoiceId): array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT sil.*, l.name as ledger_name FROM sales_invoice_lines sil LEFT JOIN ledgers l ON l.id = sil.ledger_id WHERE sil.sales_invoice_id = ? ORDER BY sil.sort_order');
    $stmt->execute([$invoiceId]);
    return $stmt->fetchAll();
}

function sales_repo_taxes(int $invoiceId): array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT * FROM sales_invoice_taxes WHERE sales_invoice_id = ?');
    $stmt->execute([$invoiceId]);
    return $stmt->fetchAll();
}

function sales_repo_next_invoice_no(int $companyId, int $fyId): string
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT COALESCE(MAX(CAST(SUBSTRING(invoice_no, 6) AS UNSIGNED)), 0) + 1 as next FROM sales_invoices WHERE company_id = ? AND financial_year_id = ?');
    $stmt->execute([$companyId, $fyId]);
    $row = $stmt->fetch();
    return 'INV-' . str_pad((string) ($row['next'] ?? 1), 5, '0', STR_PAD_LEFT);
}

function sales_repo_create(array $data): int
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO sales_invoices (company_id, financial_year_id, invoice_no, invoice_date, customer_ledger_id, subtotal, discount_amount, charges_amount, tax_amount, grand_total, narration, status, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([
        $data['company_id'],
        $data['financial_year_id'],
        $data['invoice_no'],
        $data['invoice_date'],
        $data['customer_ledger_id'],
        $data['subtotal'],
        $data['discount_amount'],
        $data['charges_amount'],
        $data['tax_amount'],
        $data['grand_total'],
        $data['narration'] ?? null,
        $data['status'] ?? 'DRAFT',
        $data['created_by'] ?? null,
    ]);
    return (int) $pdo->lastInsertId();
}

function sales_repo_create_line(int $invoiceId, array $line): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO sales_invoice_lines (sales_invoice_id, description, quantity, rate, amount, ledger_id, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([
        $invoiceId,
        $line['description'] ?? '',
        $line['quantity'] ?? 1,
        $line['rate'] ?? 0,
        $line['amount'] ?? 0,
        $line['ledger_id'] ?? null,
        $line['sort_order'] ?? 0,
    ]);
}

function sales_repo_update_voucher(int $invoiceId, int $voucherId): void
{
    $pdo = db();
    $pdo->prepare('UPDATE sales_invoices SET voucher_id = ?, status = ? WHERE id = ?')->execute([$voucherId, 'POSTED', $invoiceId]);
}

function open_refs_repo_create_ar(int $companyId, int $partyLedgerId, string $refModule, int $refId, string $refDate, string $refNumber, float $refAmount): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO open_refs (company_id, ref_type, ref_module, ref_id, party_ledger_id, ref_date, ref_number, ref_amount, open_amount, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([$companyId, 'AR_INVOICE', $refModule, $refId, $partyLedgerId, $refDate, $refNumber, $refAmount, $refAmount, 'OPEN']);
}
