<?php

declare(strict_types=1);

require_once __DIR__ . '/sales_repo.php';
require_once __DIR__ . '/../accounts/posting_engine.php';

function sales_service_post_invoice(int $invoiceId): int
{
    $invoice = sales_repo_find($invoiceId);
    if (!$invoice) {
        throw new InvalidArgumentException('Invoice not found.');
    }
    if ($invoice['status'] === 'POSTED') {
        throw new InvalidArgumentException('Invoice already posted.');
    }

    $lines = sales_service_build_voucher_lines($invoice);

    $voucherId = post_voucher(
        (int) $invoice['company_id'],
        (int) $invoice['financial_year_id'],
        'SALES',
        $invoice['invoice_date'],
        'Sales Invoice ' . $invoice['invoice_no'],
        $lines,
        'sales/invoice',
        $invoiceId
    );

    sales_repo_update_voucher($invoiceId, $voucherId);
    open_refs_repo_create_ar(
        (int) $invoice['company_id'],
        (int) $invoice['customer_ledger_id'],
        'sales_invoice',
        $invoiceId,
        $invoice['invoice_date'],
        $invoice['invoice_no'],
        (float) $invoice['grand_total']
    );

    return $voucherId;
}

function sales_service_build_voucher_lines(array $invoice): array
{
    $invoiceLines = sales_repo_lines((int) $invoice['id']);
    $taxLines = sales_repo_taxes((int) $invoice['id']);

    $lines = [];
    $grandTotal = (float) $invoice['grand_total'];
    $subtotal = (float) $invoice['subtotal'];
    $discount = (float) $invoice['discount_amount'];
    $charges = (float) $invoice['charges_amount'];
    $salesCredit = $subtotal - $discount;

    $lines[] = [
        'ledger_id' => $invoice['customer_ledger_id'],
        'debit' => $grandTotal,
        'credit' => 0,
        'narration' => 'Sales Invoice ' . $invoice['invoice_no'],
    ];

    foreach ($invoiceLines as $il) {
        $ledgerId = $il['ledger_id'] ?? null;
        if (!$ledgerId) continue;
        $amount = (float) $il['amount'];
        $lines[] = ['ledger_id' => $ledgerId, 'debit' => 0, 'credit' => $amount, 'narration' => $il['description'] ?? ''];
        $salesCredit -= $amount;
    }
    if ($salesCredit > 0) {
        $defaultSalesLedger = sales_service_get_default_sales_ledger((int) $invoice['company_id']);
        if ($defaultSalesLedger) {
            $lines[] = ['ledger_id' => $defaultSalesLedger, 'debit' => 0, 'credit' => $salesCredit, 'narration' => 'Sales'];
        }
    }

    foreach ($taxLines as $t) {
        $ledgerId = $t['ledger_id'] ?? null;
        if ($ledgerId && (float) $t['amount'] > 0) {
            $lines[] = ['ledger_id' => $ledgerId, 'debit' => 0, 'credit' => (float) $t['amount'], 'narration' => 'Output Tax'];
        }
    }

    if ($charges > 0) {
        $chargesLedger = sales_service_get_charges_ledger((int) $invoice['company_id']);
        if ($chargesLedger) {
            $lines[] = ['ledger_id' => $chargesLedger, 'debit' => 0, 'credit' => $charges, 'narration' => 'Charges'];
        }
    }

    return $lines;
}

function sales_service_get_default_sales_ledger(int $companyId): ?int
{
    $pdo = db();
    $stmt = $pdo->prepare("SELECT id FROM ledgers WHERE company_id = ? AND is_active = 1 AND group_id IN (SELECT id FROM account_groups WHERE company_id = ? AND nature = 'INCOME') LIMIT 1");
    $stmt->execute([$companyId, $companyId]);
    $row = $stmt->fetch();
    return $row ? (int) $row['id'] : null;
}

function sales_service_get_charges_ledger(int $companyId): ?int
{
    $pdo = db();
    $stmt = $pdo->prepare("SELECT id FROM ledgers WHERE company_id = ? AND is_active = 1 LIMIT 1");
    $stmt->execute([$companyId]);
    $row = $stmt->fetch();
    return $row ? (int) $row['id'] : null;
}
